"use client";
import { useState, useRef, useEffect } from "react";
import {
    Sun, Moon, Grid, X, Camera, Plus, LogOut,
    ChevronDown, ChevronUp, Home, Check, ChevronRight, Loader2
} from "lucide-react";
import { cn } from "@/lib/utils";
import Link from "next/link";
import { useAuth } from "@/context/AuthContext"
import { useRouter } from 'next/navigation';
// import { authService } from "@/lib/auth"

interface Account {
    email: string;
    name: string;
    image?: string;
    isLoggedIn?: boolean;
    user_id?: string;
}

interface ApiAccount {
    user_id: string;
    identifier: string;
    active: boolean;
}

interface AccountSwitcherProps {
    primaryAccount: Account;
    theme: string;
    toggleTheme: () => void;
}

export default function AccountSwitcher({ primaryAccount, theme, toggleTheme }: AccountSwitcherProps) {
    const [isOpen, setIsOpen] = useState(false);
    const [showMore, setShowMore] = useState(true);
    const [mergedAccounts, setMergedAccounts] = useState<Account[]>([]);
    const [fetchingAccounts, setFetchingAccounts] = useState(false);

    const menuRef = useRef<HTMLDivElement>(null);
    const { user, loading, logout } = useAuth();
    const router = useRouter();

    const handleLogOut = () => {
        logout();
        setIsOpen(false);
    };

    const validateDevice = () => {
        const deviceId = localStorage.getItem("device_id");

        if (!deviceId) {
            // No device_id found, force authentication
            router.push("/login");
            return null;
        }

        return deviceId;
    };

    // Run the check on mount
    useEffect(() => {
        validateDevice();
    }, []);

    const handleSwitchAccount = async (userId: string) => {
        const device_id = validateDevice()
        try {
            // Call your backend switch-account endpoint
            const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/switch-account/`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Device-Id': device_id || "",
                },
                credentials:"include",
                body: JSON.stringify({ user_id: userId })
            });
            if (response.ok) {
                // Refresh page or update auth context to reflect the new active user
                window.location.reload();
            }
        } catch (error) {
            console.error("Failed to switch account", error);
        }
    };

    useEffect(() => {
        const syncAccounts = async () => {
            const device_id = validateDevice()
            setFetchingAccounts(true);
            try {
                // 1. Fetch Active Sessions from API
                const res = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/accounts/`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Device-Id': device_id || "",
                    },

                });
                const apiAccounts: ApiAccount[] = await res.json();
                console.log(apiAccounts)
                // 2. Get Saved Accounts from LocalStorage
                const saved = localStorage.getItem("saved_accounts");
                let localAccounts: Account[] = saved ? JSON.parse(saved) : [];

                // 3. Merge Logic
                // Filter out the currently active user from the "Other" list
                const otherApiAccounts = apiAccounts
                    .filter(api => !api.active)
                    .map(api => ({
                        email: api.identifier,
                        name: api.identifier.split('@')[0], // Fallback name
                        user_id: api.user_id,
                        isLoggedIn: true
                    }));

                // Filter local accounts: 
                // - Must not be the current primary user
                // - Must not already be in the API list (avoid duplicates)
                const filteredLocal = localAccounts.filter(local =>
                    local.email !== user?.primary_email &&
                    !apiAccounts.some(api => api.identifier === local.email)
                ).map(local => ({ ...local, isLoggedIn: false }));

                setMergedAccounts([...otherApiAccounts, ...filteredLocal]);
            } catch (e) {
                console.error("Error syncing accounts", e);
            } finally {
                setFetchingAccounts(false);
            }
        };

        if (isOpen && user) {
            syncAccounts();
        }
    }, [isOpen, user]);

    // Close when clicking outside
    useEffect(() => {
        const handler = (e: MouseEvent) => {
            if (menuRef.current && !menuRef.current.contains(e.target as Node)) setIsOpen(false);
        };
        document.addEventListener("mousedown", handler);
        return () => document.removeEventListener("mousedown", handler);
    }, []);

    const getInitial = (name: string) => name?.charAt(0).toUpperCase() || "?";

    return (
        <div className="flex items-center gap-1 relative" ref={menuRef}>
            {/* Main Avatar Toggle */}
            {(loading) ? (
                <div className="ml-2 w-8 h-8 rounded-full flex items-center justify-center">
                    <Loader2 className="h-5 w-5 animate-spin text-sidebar-primary" />
                </div>
            ) : (
                    <div
                        onClick={() => setIsOpen(!isOpen)}
                        className="ml-2 w-8 h-8 rounded-full bg-sidebar-primary flex items-center justify-center text-sm text-white font-medium cursor-pointer border border-border shadow-sm hover:ring-4 hover:ring-sidebar-primary/20 transition-all overflow-hidden"
                    >
                        {user?.profile_picture ? (
                            <img src={user.profile_picture} alt="Profile" className="w-full h-full object-cover" />
                        ) : (
                                user?.first_name?.charAt(0).toUpperCase()
                            )}
                    </div>
                )}

            {isOpen && user && (
                <div className="absolute overflow-y-auto max-h-[80vh] top-12 right-0 w-[350px] bg-[#f0f4f9] dark:bg-[#1e1f20] rounded-[28px] shadow-2xl z-[100] border border-border/50 p-2 animate-in fade-in zoom-in-95 duration-200">

                    {/* Primary Account Card */}
                    <div className="bg-card rounded-[24px] p-4 text-center relative shadow-sm">
                        <button onClick={() => setIsOpen(false)} className="absolute right-4 top-4 text-muted-foreground hover:bg-accent p-1.5 rounded-full transition-colors">
                            <X size={18} />
                        </button>

                        <p className="text-sm font-medium mb-4 text-foreground/80">{user?.primary_email}</p>

                        <div className="relative w-20 h-20 mx-auto group">
                            <div className="w-full h-full rounded-full bg-green-700 flex items-center justify-center text-3xl text-white overflow-hidden">
                                {user?.profile_picture ? (
                                    <img src={user.profile_picture} alt="Profile" className="w-full h-full object-cover" />
                                ) : (
                                        user?.first_name?.charAt(0).toUpperCase()
                                    )}
                            </div>
                            <Link href="/settings" className="absolute bottom-0 right-0 p-1.5 bg-card border border-border rounded-full shadow-md hover:bg-accent">
                                <Camera size={14} />
                            </Link>
                        </div>

                        <h2 className="text-xl font-normal mt-3 text-foreground">Hi, {user?.first_name}!</h2>

                        <Link href="/settings">
                            <button className="mt-4 px-6 py-2 border border-border rounded-full text-sm font-semibold hover:bg-sidebar-primary/5 transition-colors text-sidebar-primary">
                                Manage your Account
                            </button>
                        </Link>
                    </div>

                    {/* Merged Accounts List */}
                    <div className="mt-2 bg-card rounded-[24px] overflow-hidden shadow-sm">
                        <button
                            onClick={() => setShowMore(!showMore)}
                            className="w-full flex items-center justify-between p-4 hover:bg-accent/50 transition-colors"
                        >
                            <span className="text-sm font-medium">Other accounts</span>
                            {showMore ? <ChevronUp size={18} /> : <ChevronDown size={18} />}
                        </button>

                        {showMore && (
                            <div className="pb-2">
                                {mergedAccounts.map((acc, i) => (
                                    <div
                                        key={acc.email}
                                        onClick={() => acc.isLoggedIn ? handleSwitchAccount(acc.user_id!) : router.push(`/login?email=${acc.email}`)}
                                        className="flex items-center justify-between px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40 group"
                                    >
                                        <div className="flex items-center gap-3">
                                            <div className="w-9 h-9 rounded-full bg-muted flex items-center justify-center text-muted-foreground text-sm font-bold overflow-hidden">
                                                {acc.image ? <img src={acc.image} alt="" /> : getInitial(acc.name)}
                                            </div>
                                            <div className="text-left">
                                                <p className="text-sm font-semibold leading-none text-foreground">{acc.name}</p>
                                                <p className="text-xs text-muted-foreground mt-1">{acc.email}</p>
                                            </div>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            {!acc.isLoggedIn && <span className="text-[10px] bg-muted px-1.5 py-0.5 rounded text-muted-foreground">Signed out</span>}
                                            <ChevronRight size={16} className="text-muted-foreground opacity-0 group-hover:opacity-100 transition-opacity" />
                                        </div>
                                    </div>
                                ))}

                                <Link href="/login" className="flex items-center gap-3 px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40">
                                    <div className="w-9 h-9 flex items-center justify-center">
                                        <Plus size={20} className="text-muted-foreground" />
                                    </div>
                                    <span className="text-sm font-medium">Add another account</span>
                                </Link>

                                <button onClick={handleLogOut} className="w-full flex items-center gap-3 px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40">
                                    <div className="w-9 h-9 flex items-center justify-center">
                                        <LogOut size={20} className="text-muted-foreground" />
                                    </div>
                                    <span className="text-sm font-medium">Sign out of all accounts</span>
                                </button>
                            </div>
                        )}
                    </div>

                    <div className="flex justify-center gap-4 py-4 text-[10px] text-muted-foreground font-medium uppercase tracking-tighter">
                        <span className="hover:underline cursor-pointer">Privacy</span>
                        <span>•</span>
                        <span className="hover:underline cursor-pointer">Terms</span>
                    </div>
                </div>
            )}
        </div>
    );
}